<?php
declare(strict_types=1);

require_once __DIR__ . '/../lib/auth.php';
require_once __DIR__ . '/../lib/db.php';
require_once __DIR__ . '/../lib/respond.php';

final class AdminUsersController {
  public static function list(): void {
    require_role(['admin']);
    $rows = db()->query(
      "SELECT u.id, u.username, u.display_name, u.role, u.org_id, u.is_active, u.last_seen_at, u.created_at,
              o.name AS org_name, o.short_name AS org_short
       FROM app_user u
       JOIN org o ON o.id=u.org_id
       ORDER BY u.is_active DESC, u.role ASC, u.username ASC"
    )->fetchAll();
    api_json(['ok'=>true, 'users'=>$rows]);
  }

  public static function create(): void {
    require_role(['admin']);
    $in = json_input();
    $username = trim((string)($in['username'] ?? ''));
    $display = trim((string)($in['display_name'] ?? ''));
    $role = (string)($in['role'] ?? 'viewer');
    $orgId = (string)($in['org_id'] ?? '');
    $password = (string)($in['password'] ?? '');
    if ($username === '' || $orgId === '' || $password === '') api_error(400, 'username, org_id, password required');
    if (!in_array($role, ['admin','el','gf','viewer'], true)) api_error(400, 'invalid role');

    // Ensure org exists
    $chk = db()->prepare('SELECT 1 FROM org WHERE id=:id');
    $chk->execute([':id'=>$orgId]);
    if (!$chk->fetchColumn()) api_error(400, 'org not found');

    $hash = password_hash($password, PASSWORD_DEFAULT);
    $stmt = db()->prepare('INSERT INTO app_user(username, display_name, role, org_id, password_hash, is_active, created_at)
      VALUES(:u, NULLIF(:d,\'\'), :r, :oid, :ph, true, now()) RETURNING id');
    $stmt->execute([':u'=>$username, ':d'=>$display, ':r'=>$role, ':oid'=>$orgId, ':ph'=>$hash]);
    $id = (int)$stmt->fetchColumn();
    EventsController::emitGlobal('users.updated', ['user_id'=>$id]);
    api_json(['ok'=>true, 'id'=>$id], 201);
  }

  public static function update(int $userId): void {
    require_role(['admin']);
    $in = json_input();
    $fields = [];
    $params = [':id'=>$userId];
    if (array_key_exists('display_name', $in)) { $fields[]='display_name=NULLIF(:d,\'\')'; $params[':d']=trim((string)$in['display_name']); }
    if (array_key_exists('role', $in)) {
      $role = (string)$in['role'];
      if (!in_array($role, ['admin','el','gf','viewer'], true)) api_error(400,'invalid role');
      $fields[]='role=:r'; $params[':r']=$role;
    }
    if (array_key_exists('org_id', $in)) {
      $orgId = (string)$in['org_id'];
      if ($orgId === '') api_error(400,'org_id required');
      $chk = db()->prepare('SELECT 1 FROM org WHERE id=:id');
      $chk->execute([':id'=>$orgId]);
      if (!$chk->fetchColumn()) api_error(400,'org not found');
      $fields[]='org_id=:oid'; $params[':oid']=$orgId;
    }
    if (array_key_exists('is_active', $in)) { $fields[]='is_active=:a'; $params[':a']=(bool)$in['is_active']; }
    if (!$fields) api_error(400, 'nothing to update');

    $sql = 'UPDATE app_user SET '.implode(',', $fields).', updated_at=now() WHERE id=:id';
    db()->prepare($sql)->execute($params);
    EventsController::emitGlobal('users.updated', ['user_id'=>$userId]);
    api_json(['ok'=>true]);
  }

  public static function resetPassword(int $userId): void {
    require_role(['admin']);
    $in = json_input();
    $password = (string)($in['password'] ?? '');
    if ($password === '') api_error(400, 'password required');
    $hash = password_hash($password, PASSWORD_DEFAULT);
    db()->prepare('UPDATE app_user SET password_hash=:ph, updated_at=now() WHERE id=:id')
      ->execute([':ph'=>$hash, ':id'=>$userId]);
    EventsController::emitGlobal('users.updated', ['user_id'=>$userId]);
    api_json(['ok'=>true]);
  }
}
