<?php
declare(strict_types=1);

final class ChatController {
  public static function list(string $incidentId): void {
    $u = require_login();
    $pdo = db();
    if ($u['role'] !== 'admin') {
      $chk = $pdo->prepare('SELECT 1 FROM incident_org WHERE incident_id=:iid AND org_id=:oid');
      $chk->execute([':iid'=>$incidentId, ':oid'=>$u['org_id']]);
      if (!$chk->fetchColumn()) api_error(403, 'not a participant');
    }

    $stmt = $pdo->prepare('SELECT m.id, m.author_user_id, m.incident_id, m.scope, m.org_id, m.to_org_id, m.to_user_id, m.message,
        m.mentions_json, m.created_at,
        u.display_name AS author_name, u.username AS author_username, o.short_name AS author_org
      FROM message m
      JOIN app_user u ON u.id=m.author_user_id
      JOIN org o ON o.id=u.org_id
      WHERE m.incident_id=:iid
      ORDER BY m.created_at DESC
      LIMIT 200');
    $stmt->execute([':iid'=>$incidentId]);
    $rows = array_reverse($stmt->fetchAll());
    foreach ($rows as &$r) {
      $r['mentions'] = json_decode($r['mentions_json'] ?? '[]', true);
      unset($r['mentions_json']);
    }

    // Scope filtering in PHP (simple)
    $out = [];
    foreach ($rows as $r) {
      if ($r['scope']==='public') { $out[] = $r; continue; }
      if ($r['scope']==='org' && $r['to_org_id']===$u['org_id']) { $out[] = $r; continue; }
      if ($r['scope']==='org' && $r['org_id']===$u['org_id']) { $out[] = $r; continue; }
      if ($r['scope']==='private' && (string)$r['to_user_id']===(string)$u['id']) { $out[] = $r; continue; }
      if ($r['scope']==='private' && (string)$r['author_user_id']===(string)$u['id']) { $out[] = $r; continue; }
      if ($r['scope']==='gfs' && in_array($u['role'], ['gf','el','admin'], true)) { $out[]=$r; continue; }
      if ($r['scope']==='all') { $out[]=$r; continue; }
    }

    api_json(['ok'=>true, 'messages'=>$out]);
  }

  public static function send(string $incidentId): void {
    $u = require_role(['admin','el','gf']);
    $in = json_input();
    $scope = (string)($in['scope'] ?? 'public'); // public|org|gfs|all|private
    $toOrgId = (string)($in['to_org_id'] ?? '');
    $toUserId = (string)($in['to_user_id'] ?? '');
    $msg = trim((string)($in['message'] ?? ''));
    $mentions = $in['mentions'] ?? [];
    if (!is_array($mentions)) $mentions = [];
    if ($msg==='') api_error(400, 'message required');

    // Validate scope
    $allowed = ['public','org','gfs','all','private'];
    if (!in_array($scope, $allowed, true)) api_error(400, 'invalid scope');

    // Default org scope: send to own org
    if ($scope==='org' && $toOrgId==='') $toOrgId = $u['org_id'];

    $pdo = db();
    $stmt = $pdo->prepare('INSERT INTO message(id, incident_id, author_user_id, scope, org_id, to_org_id, to_user_id, message, mentions_json)
      VALUES (gen_random_uuid(), :iid, :uid, :sc, :oid, NULLIF(:to_oid,\'\')::uuid, NULLIF(:to_uid,\'\')::int, :m, :men::jsonb)
      RETURNING id');
    $stmt->execute([
      ':iid'=>$incidentId, ':uid'=>$u['id'], ':sc'=>$scope, ':oid'=>$u['org_id'],
      ':to_oid'=>$toOrgId, ':to_uid'=>$toUserId,
      ':m'=>$msg,
      ':men'=>json_encode($mentions, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES),
    ]);
    $id = (string)$stmt->fetchColumn();

    // Emit events depending on scope
    if ($scope==='public' || $scope==='all' || $scope==='gfs') {
      EventsController::emitIncident($incidentId, null, 'all', 'chat.updated', ['incident_id'=>$incidentId]);
    } elseif ($scope==='org') {
      EventsController::emitIncident($incidentId, $toOrgId, 'org', 'chat.updated', ['incident_id'=>$incidentId]);
      if ($toOrgId !== $u['org_id']) {
        EventsController::emitIncident($incidentId, $u['org_id'], 'org', 'chat.updated', ['incident_id'=>$incidentId]);
      }
    } elseif ($scope==='private') {
      EventsController::emitIncident($incidentId, $u['org_id'], 'org', 'chat.updated', ['incident_id'=>$incidentId]);
    }

    api_json(['ok'=>true, 'id'=>$id], 201);
  }
}
