<?php
declare(strict_types=1);

final class MembersController {
  private static function memberCapabilities(PDO $pdo, string $memberId): array {
    $stmt = $pdo->prepare('SELECT c.key
      FROM member_capability mc
      JOIN capability c ON c.id = mc.capability_id
      WHERE mc.member_id=:mid AND c.is_active=true
      ORDER BY c.sort_order ASC, c.label ASC');
    $stmt->execute([':mid'=>$memberId]);
    return $stmt->fetchAll(PDO::FETCH_COLUMN) ?: [];
  }

  private static function syncMemberCapabilities(PDO $pdo, string $memberId, string $orgId, array $keys): void {
    // normalize keys
    $clean = [];
    foreach ($keys as $k) {
      if (!is_string($k)) continue;
      $k = strtolower(trim($k));
      if ($k === '') continue;
      if (!preg_match('/^[a-z0-9_\-\.]{2,40}$/', $k)) continue;
      $clean[$k] = true;
    }
    $keys = array_keys($clean);

    // Ensure all referenced capability keys exist (org-specific if present, fallback to global)
    $capIds = [];
    if (count($keys)) {
      $placeholders = implode(',', array_fill(0, count($keys), '?'));
      $q = "SELECT id, key, org_id
            FROM capability
            WHERE is_active=true
              AND key IN ($placeholders)
              AND (org_id IS NULL OR org_id = ?)
            ORDER BY org_id NULLS LAST"; // org-specific first, global last
      $stmt = $pdo->prepare($q);
      $params = array_merge($keys, [$orgId]);
      $stmt->execute($params);
      $rows = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];

      // pick first occurrence per key (org-specific preferred due to ORDER BY)
      foreach ($rows as $r) {
        $k = (string)$r['key'];
        if (!isset($capIds[$k])) $capIds[$k] = (int)$r['id'];
      }
    }

    // Auto-create missing keys as org-specific capabilities (keeps system flexible, avoids hardcoding)
    $missing = [];
    foreach ($keys as $k) {
      if (!isset($capIds[$k])) $missing[] = $k;
    }
    if (count($missing)) {
      $ins = $pdo->prepare('INSERT INTO capability(org_id, key, label, sort_order, is_active)
        VALUES(:oid, :k, :l, 200, true)
        ON CONFLICT (org_id, key) DO NOTHING');
      foreach ($missing as $k) {
        $ins->execute([':oid'=>$orgId, ':k'=>$k, ':l'=>strtoupper($k)]);
      }

      // Re-fetch to resolve ids (again using IN placeholders)
      $placeholders = implode(',', array_fill(0, count($keys), '?'));
      $stmt = $pdo->prepare("SELECT id, key, org_id
        FROM capability
        WHERE is_active=true
          AND key IN ($placeholders)
          AND (org_id IS NULL OR org_id = ?)
        ORDER BY org_id NULLS LAST");
      $params = array_merge($keys, [$orgId]);
      $stmt->execute($params);
      $rows = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];

      foreach ($rows as $r) {
        $k = (string)$r['key'];
        if (!isset($capIds[$k])) $capIds[$k] = (int)$r['id'];
      }
    }

    $pdo->prepare('DELETE FROM member_capability WHERE member_id=:mid')->execute([':mid'=>$memberId]);
    if (count($capIds)) {
      $ins = $pdo->prepare('INSERT INTO member_capability(member_id, capability_id) VALUES(:mid, :cid)');
      foreach ($capIds as $cid) {
        $ins->execute([':mid'=>$memberId, ':cid'=>$cid]);
      }
    }

    // keep denormalized json for convenience
    $pdo->prepare('UPDATE member SET capabilities_json=:caps::jsonb, updated_at=now() WHERE id=:mid')
      ->execute([':caps'=>json_encode(array_keys($capIds), JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES), ':mid'=>$memberId]);
  }

  public static function listDirectory(): void {
    $u = require_login();
    $pdo = db();
    $orgId = (string)($_GET['org_id'] ?? $u['org_id']);
    if ($u['role'] !== 'admin' && $orgId !== $u['org_id']) {
      api_error(403, 'Cannot list other org directory');
    }
    $stmt = $pdo->prepare('SELECT id, org_id, display_name, short_code, is_active, created_at, updated_at
      FROM member WHERE org_id=:oid ORDER BY display_name');
    $stmt->execute([':oid'=>$orgId]);
    $rows = $stmt->fetchAll();
    foreach ($rows as &$r) {
      $r['capabilities'] = self::memberCapabilities($pdo, (string)$r['id']);
    }
    api_json(['ok'=>true, 'members'=>$rows]);
  }

  public static function upsertMember(): void {
    $u = require_role(['admin','el','gf']);
    $in = json_input();
    $id = (string)($in['id'] ?? '');
    $orgId = (string)($in['org_id'] ?? $u['org_id']);
    if ($u['role'] !== 'admin' && $orgId !== $u['org_id']) api_error(403, 'Cannot modify other org members');

    $name = trim((string)($in['display_name'] ?? ''));
    if ($name === '') api_error(400, 'display_name required');
    $code = trim((string)($in['short_code'] ?? ''));
    $caps = $in['capabilities'] ?? [];
    if (!is_array($caps)) $caps = [];

    if ($id && !preg_match('/^[a-f0-9\-]{36}$/', $id)) api_error(400, 'invalid id');

    if (!$id) {
      $stmt = db()->prepare('INSERT INTO member(id, org_id, display_name, short_code, capabilities_json, is_active)
        VALUES (gen_random_uuid(), :oid, :n, :c, :caps::jsonb, true) RETURNING id');
      $stmt->execute([':oid'=>$orgId, ':n'=>$name, ':c'=>$code, ':caps'=>json_encode($caps, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES)]);
      $id = (string)$stmt->fetchColumn();
      // sync normalized capabilities
      self::syncMemberCapabilities(db(), $id, $orgId, $caps);
      EventsController::emitGlobal("member.created", ["org_id"=>$orgId, "member_id"=>$id]);
      api_json(['ok'=>true, 'id'=>$id], 201);
    }

    db_tx(function(PDO $pdo) use ($id, $orgId, $name, $code, $caps) {
      $pdo->prepare('UPDATE member SET display_name=:n, short_code=:c, updated_at=now() WHERE id=:id AND org_id=:oid')
        ->execute([':n'=>$name, ':c'=>$code, ':id'=>$id, ':oid'=>$orgId]);
      self::syncMemberCapabilities($pdo, $id, $orgId, $caps);
    });

    api_json(['ok'=>true, 'id'=>$id]);
  }

  public static function deactivate(string $id): void {
    $u = require_role(['admin','el']);
    db()->prepare('UPDATE member SET is_active=false, updated_at=now() WHERE id=:id AND org_id=:oid')
      ->execute([':id'=>$id, ':oid'=>$u['org_id']]);
    api_json(['ok'=>true]);
  }

  public static function listParticipants(string $incidentId): void {
    $u = require_login();
    $pdo = db();
    // Ensure user participates
    if ($u['role'] !== 'admin') {
      $chk = $pdo->prepare('SELECT 1 FROM incident_org WHERE incident_id=:iid AND org_id=:oid');
      $chk->execute([':iid'=>$incidentId, ':oid'=>$u['org_id']]);
      if (!$chk->fetchColumn()) api_error(403, 'not a participant of this incident');
    }

    $privacy = public_settings()['privacy'];
    $showNamesOther = (bool)($privacy['showNamesToOtherOrgs'] ?? false);
    $pseudoOther = (bool)($privacy['pseudonymizeOtherOrgs'] ?? true);

    $stmt = $pdo->prepare("SELECT ci.id AS checkin_id, ci.status, ci.checked_in_at, ci.checked_out_at, ci.checkout_reason,
        m.id AS member_id, m.org_id, m.display_name, m.short_code,
        o.name AS org_name, o.short_name AS org_short, o.color AS org_color
      FROM checkin ci
      JOIN member m ON m.id = ci.member_id
      JOIN org o ON o.id = m.org_id
      WHERE ci.incident_id=:iid
      ORDER BY o.name, m.display_name");
    $stmt->execute([':iid'=>$incidentId]);
    $rows = $stmt->fetchAll();

    foreach ($rows as &$r) {
      $r['capabilities'] = self::memberCapabilities($pdo, (string)$r['member_id']);

      // loan info (if member is loaned to another org, or loaned in)
      $loanStmt = $pdo->prepare("SELECT from_org_id, to_org_id, status FROM member_loan WHERE incident_id=:iid AND member_id=:mid AND status='active' LIMIT 1");
      $loanStmt->execute([':iid'=>$incidentId, ':mid'=>$r['member_id']]);
      $loan = $loanStmt->fetch();
      if ($loan) {
        $r['loan'] = [
          'from_org_id' => $loan['from_org_id'],
          'to_org_id' => $loan['to_org_id'],
          'status' => $loan['status'],
        ];
      } else {
        $r['loan'] = null;
      }

      $isOwnOrg = ($r['org_id'] === $u['org_id']) || $u['role']==='admin' || $u['role']==='el';
      if (!$isOwnOrg) {
        if (!$showNamesOther) {
          // remove display name
          $r['display_name'] = null;
        }
        if ($pseudoOther) {
          // keep stable pseudonym per incident+member (hash of ids)
          $r['pseudonym'] = 'P-' . substr(hash('sha256', $incidentId . ':' . $r['member_id']), 0, 6);
        }
      }
    }

    api_json(['ok'=>true, 'participants'=>$rows]);
  }

  /**
   * Roster of an org for an incident: includes all members of org plus members loaned-in,
   * with check-in status.
   */
  public static function orgRoster(string $incidentId): void {
    $u = require_login();
    $pdo = db();
    $orgId = (string)($_GET['org_id'] ?? $u['org_id']);
    if ($u['role'] !== 'admin' && $orgId !== $u['org_id']) api_error(403, 'Cannot view other org roster');

    // must be participant (unless admin)
    if ($u['role'] !== 'admin') {
      $chk = $pdo->prepare('SELECT 1 FROM incident_org WHERE incident_id=:iid AND org_id=:oid');
      $chk->execute([':iid'=>$incidentId, ':oid'=>$orgId]);
      if (!$chk->fetchColumn()) api_error(403, 'not a participant of this incident');
    }

    // local members + loaned-in members (active)
    $stmt = $pdo->prepare("WITH roster AS (
        SELECT m.id AS member_id, m.org_id, m.display_name, m.short_code, false AS loaned_in
        FROM member m
        WHERE m.org_id=:oid AND m.is_active=true
        UNION
        SELECT m.id AS member_id, m.org_id, m.display_name, m.short_code, true AS loaned_in
        FROM member_loan ml
        JOIN member m ON m.id = ml.member_id
        WHERE ml.incident_id=:iid AND ml.to_org_id=:oid AND ml.status='active'
      )
      SELECT r.member_id, r.org_id, r.display_name, r.short_code, r.loaned_in,
             ci.status, ci.checked_in_at, ci.checked_out_at, ci.checkout_reason
      FROM roster r
      LEFT JOIN checkin ci ON ci.incident_id=:iid AND ci.member_id=r.member_id
      ORDER BY r.loaned_in DESC, r.display_name ASC");
    $stmt->execute([':oid'=>$orgId, ':iid'=>$incidentId]);
    $rows = $stmt->fetchAll();
    foreach ($rows as &$r) {
      $r['capabilities'] = self::memberCapabilities($pdo, (string)$r['member_id']);
      if (!$r['status']) {
        $r['status'] = 'out';
      }
    }
    api_json(['ok'=>true, 'roster'=>$rows]);
  }

  public static function checkIn(string $incidentId): void {
    $u = require_role(['admin','el','gf']);
    $in = json_input();
    $memberIds = $in['member_ids'] ?? [];
    if (!is_array($memberIds) || count($memberIds) === 0) api_error(400, 'member_ids required');

    db_tx(function(PDO $pdo) use ($incidentId, $memberIds, $u) {
      $stmt = $pdo->prepare('INSERT INTO checkin(id, incident_id, member_id, status, checked_in_at, created_by_user_id)
        VALUES (gen_random_uuid(), :iid, :mid, :st, now(), :uid)
        ON CONFLICT (incident_id, member_id) DO UPDATE SET status=EXCLUDED.status, checked_in_at=EXCLUDED.checked_in_at, checked_out_at=NULL, checkout_reason=NULL, updated_at=now()');
      foreach ($memberIds as $mid) {
        if (!is_string($mid) || !preg_match('/^[a-f0-9\-]{36}$/', $mid)) continue;
        // Ensure member belongs to org or is loaned to org (unless admin)
        if ($u['role'] !== 'admin') {
          $own = $pdo->prepare('SELECT 1 FROM member WHERE id=:id AND org_id=:oid AND is_active=true');
          $own->execute([':id'=>$mid, ':oid'=>$u['org_id']]);
          $ok = (bool)$own->fetchColumn();
          if (!$ok) {
            $loan = $pdo->prepare("SELECT 1 FROM member_loan WHERE incident_id=:iid AND member_id=:mid AND to_org_id=:to AND status='active'");
            $loan->execute([':iid'=>$incidentId, ':mid'=>$mid, ':to'=>$u['org_id']]);
            $ok = (bool)$loan->fetchColumn();
          }
          if (!$ok) continue;
        }
        $stmt->execute([':iid'=>$incidentId, ':mid'=>$mid, ':st'=>'in', ':uid'=>$u['id']]);
      }
    });

    EventsController::emitIncident($incidentId, null, 'all', 'participants.updated', ['incident_id'=>$incidentId]);
    api_json(['ok'=>true]);
  }

  /**
   * Create a member (if needed) and check-in immediately.
   * Useful for quick participant onboarding.
   */
  public static function checkInQuick(string $incidentId): void {
    $u = require_role(['admin','el','gf']);
    $in = json_input();
    $orgId = (string)($in['org_id'] ?? $u['org_id']);
    if ($u['role'] !== 'admin' && $orgId !== $u['org_id']) api_error(403, 'Cannot create for other org');

    $name = trim((string)($in['display_name'] ?? ''));
    if ($name === '') api_error(400, 'display_name required');
    $code = trim((string)($in['short_code'] ?? ''));
    $caps = $in['capabilities'] ?? [];
    if (!is_array($caps)) $caps = [];

    $newId = null;
    db_tx(function(PDO $pdo) use ($incidentId, $u, $orgId, $name, $code, $caps, &$newId) {
      $stmt = $pdo->prepare('INSERT INTO member(id, org_id, display_name, short_code, capabilities_json, is_active)
        VALUES (gen_random_uuid(), :oid, :n, :c, :caps::jsonb, true) RETURNING id');
      $stmt->execute([':oid'=>$orgId, ':n'=>$name, ':c'=>$code, ':caps'=>json_encode($caps, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES)]);
      $newId = (string)$stmt->fetchColumn();
      self::syncMemberCapabilities($pdo, $newId, $orgId, $caps);

      $pdo->prepare('INSERT INTO checkin(id, incident_id, member_id, status, checked_in_at, created_by_user_id)
        VALUES (gen_random_uuid(), :iid, :mid, :st, now(), :uid)
        ON CONFLICT (incident_id, member_id) DO UPDATE SET status=EXCLUDED.status, checked_in_at=EXCLUDED.checked_in_at, checked_out_at=NULL, checkout_reason=NULL, updated_at=now()')
        ->execute([':iid'=>$incidentId, ':mid'=>$newId, ':st'=>'in', ':uid'=>$u['id']]);
    });

    EventsController::emitGlobal("member.created", ["org_id"=>$orgId, "member_id"=>$newId]);
    EventsController::emitIncident($incidentId, null, 'all', 'participants.updated', ['incident_id'=>$incidentId]);
    api_json(['ok'=>true, 'member_id'=>$newId], 201);
  }

  public static function checkOut(string $incidentId): void {
    $u = require_role(['admin','el','gf']);
    $in = json_input();
    $reason = trim((string)($in['reason'] ?? ''));
    $memberId = (string)($in['member_id'] ?? '');
    $all = (bool)($in['all'] ?? false);

    $pdo = db();
    if ($all) {
      // Check out everyone (EL/admin only)
      if (!in_array($u['role'], ['admin','el'], true)) api_error(403, 'Only EL/Admin can mass-checkout');
      $stmt = $pdo->prepare('UPDATE checkin SET status=:st, checked_out_at=now(), checkout_reason=:r, updated_at=now()
        WHERE incident_id=:iid AND status<>:st');
      $stmt->execute([':st'=>'out', ':r'=>$reason, ':iid'=>$incidentId]);
    } else {
      if (!preg_match('/^[a-f0-9\-]{36}$/', $memberId)) api_error(400, 'member_id required');
      // Org scope: GF may only checkout own org members (unless admin/el)
      if (!in_array($u['role'], ['admin','el'], true)) {
        $chk = $pdo->prepare('SELECT 1 FROM checkin ci JOIN member m ON m.id=ci.member_id WHERE ci.incident_id=:iid AND ci.member_id=:mid AND m.org_id=:oid');
        $chk->execute([':iid'=>$incidentId, ':mid'=>$memberId, ':oid'=>$u['org_id']]);
        if (!$chk->fetchColumn()) api_error(403, 'Not allowed');
      }
      $stmt = $pdo->prepare('UPDATE checkin SET status=:st, checked_out_at=now(), checkout_reason=:r, updated_at=now()
        WHERE incident_id=:iid AND member_id=:mid');
      $stmt->execute([':st'=>'out', ':r'=>$reason, ':iid'=>$incidentId, ':mid'=>$memberId]);
    }

    EventsController::emitIncident($incidentId, null, 'all', 'participants.updated', ['incident_id'=>$incidentId]);
    api_json(['ok'=>true]);
  }
}
