<?php
declare(strict_types=1);

final class TeamsController {
  public static function list(string $incidentId): void {
    $u = require_login();
    $pdo = db();
    if ($u['role'] !== 'admin') {
      $chk = $pdo->prepare('SELECT 1 FROM incident_org WHERE incident_id=:iid AND org_id=:oid');
      $chk->execute([':iid'=>$incidentId, ':oid'=>$u['org_id']]);
      if (!$chk->fetchColumn()) api_error(403, 'not a participant');
    }

    $stmt = $pdo->prepare('SELECT t.id, t.incident_id, t.org_id, t.name, t.callsign, t.status, t.last_contact_at, t.created_at, o.name AS org_name, o.color AS org_color
      FROM team t
      JOIN org o ON o.id=t.org_id
      WHERE t.incident_id=:iid
      ORDER BY o.name, t.name');
    $stmt->execute([':iid'=>$incidentId]);
    $teams = $stmt->fetchAll();

    $ids = array_map(fn($r) => $r['id'], $teams);
    $byTeam = [];
    if (count($ids) > 0) {
      // PDO doesn't support array binding natively; use string list safely
      $in = "('" . implode("','", array_map(fn($x)=>str_replace("'","''",$x), $ids)) . "')";
      $rows = $pdo->query('SELECT tm.team_id, tm.role_in_team, tm.is_lead, m.id AS member_id, m.org_id, m.display_name, m.short_code, m.capabilities_json, o.name AS org_name, o.color AS org_color
        FROM team_member tm
        JOIN member m ON m.id=tm.member_id
        JOIN org o ON o.id=m.org_id
        WHERE tm.team_id IN ' . $in)->fetchAll();
      foreach ($rows as $r) {
        $r['capabilities'] = json_decode($r['capabilities_json'] ?? '[]', true);
        unset($r['capabilities_json']);
        $byTeam[$r['team_id']][] = $r;
      }
    }

    foreach ($teams as &$t) {
      $t['members'] = $byTeam[$t['id']] ?? [];
    }

    api_json(['ok'=>true, 'teams'=>$teams]);
  }

  public static function create(string $incidentId): void {
    $u = require_role(['admin','el','gf']);
    $in = json_input();
    $name = trim((string)($in['name'] ?? ''));
    if ($name === '') api_error(400, 'name required');
    $callsign = trim((string)($in['callsign'] ?? ''));
    $orgId = (string)($in['org_id'] ?? $u['org_id']);
    if (!in_array($u['role'], ['admin','el'], true) && $orgId !== $u['org_id']) api_error(403, 'cannot create team for other org');

    $stmt = db()->prepare('INSERT INTO team(id, incident_id, org_id, name, callsign, status, created_by_user_id)
      VALUES (gen_random_uuid(), :iid, :oid, :n, :c, :st, :uid) RETURNING id');
    $stmt->execute([':iid'=>$incidentId, ':oid'=>$orgId, ':n'=>$name, ':c'=>$callsign, ':st'=>'available', ':uid'=>$u['id']]);
    $id = (string)$stmt->fetchColumn();

    db()->prepare('INSERT INTO audit_log(id, incident_id, actor_user_id, action, entity_type, entity_id, meta_json)
      VALUES (gen_random_uuid(), :iid, :uid, :act, :et, :eid, :m::jsonb)')
      ->execute([':iid'=>$incidentId, ':uid'=>$u['id'], ':act'=>'team.create', ':et'=>'team', ':eid'=>$id, ':m'=>json_encode(['name'=>$name, 'org_id'=>$orgId])]);

    EventsController::emitIncident($incidentId, null, 'all', 'teams.updated', ['incident_id'=>$incidentId]);
    api_json(['ok'=>true, 'id'=>$id], 201);
  }

  public static function update(string $teamId): void {
    $u = require_role(['admin','el','gf']);
    $in = json_input();
    $pdo = db();
    $t = $pdo->prepare('SELECT id, incident_id, org_id FROM team WHERE id=:id');
    $t->execute([':id'=>$teamId]);
    $team = $t->fetch();
    if (!$team) api_error(404, 'team not found');
    if (!in_array($u['role'], ['admin','el'], true) && $team['org_id'] !== $u['org_id']) api_error(403, 'not allowed');

    $fields = [];
    $params = [':id'=>$teamId];
    if (isset($in['name'])) { $fields[]='name=:n'; $params[':n']=trim((string)$in['name']); }
    if (isset($in['callsign'])) { $fields[]='callsign=:c'; $params[':c']=trim((string)$in['callsign']); }
    if (isset($in['status'])) { $fields[]='status=:st'; $params[':st']=(string)$in['status']; }
    if (isset($in['last_contact_at'])) { $fields[]='last_contact_at=:lca'; $params[':lca']=(string)$in['last_contact_at']; }
    if (!$fields) api_error(400, 'nothing to update');

    $sql = 'UPDATE team SET ' . implode(',', $fields) . ', updated_at=now() WHERE id=:id';
    $pdo->prepare($sql)->execute($params);

    EventsController::emitIncident($team['incident_id'], null, 'all', 'teams.updated', ['incident_id'=>$team['incident_id']]);
    api_json(['ok'=>true]);
  }

  public static function assignMembers(string $teamId): void {
    $u = require_role(['admin','el','gf']);
    $in = json_input();
    // Accept either legacy {member_ids:[uuid,...]} or modern {members:[{member_id, role, is_lead}, ...]}
    $memberIds = $in['member_ids'] ?? null;
    $members = $in['members'] ?? null;
    if ($memberIds !== null && !is_array($memberIds)) api_error(400, 'member_ids must be array');
    if ($members !== null && !is_array($members)) api_error(400, 'members must be array');

    $pdo = db();
    $t = $pdo->prepare('SELECT id, incident_id, org_id FROM team WHERE id=:id');
    $t->execute([':id'=>$teamId]);
    $team = $t->fetch();
    if (!$team) api_error(404, 'team not found');
    if (!in_array($u['role'], ['admin','el'], true) && $team['org_id'] !== $u['org_id']) api_error(403, 'not allowed');

    // Determine default role from settings
    $teamSettings = setting('team', []);
    $roles = $teamSettings['roles'] ?? [];
    $defaultRole = 'member';
    if (is_array($roles) && count($roles) > 0) {
      $first = $roles[0];
      if (is_array($first) && !empty($first['key'])) $defaultRole = (string)$first['key'];
    }

    // Normalize payload
    $normalized = [];
    if (is_array($members)) {
      foreach ($members as $m) {
        if (!is_array($m)) continue;
        $mid = (string)($m['member_id'] ?? '');
        if (!preg_match('/^[a-f0-9\-]{36}$/', $mid)) continue;

        $role = trim((string)($m['role'] ?? ''));
        if ($role === '') $role = $defaultRole;

        // IMPORTANT: default false if missing; handle various truthy values
        $isLeadRaw = $m['is_lead'] ?? false;
        $isLead = false;
        if (is_bool($isLeadRaw)) {
          $isLead = $isLeadRaw;
        } elseif (is_numeric($isLeadRaw)) {
          $isLead = ((int)$isLeadRaw) === 1;
        } elseif (is_string($isLeadRaw)) {
          $v = strtolower(trim($isLeadRaw));
          $isLead = in_array($v, ['1','true','yes','y','on'], true);
        }

        $normalized[] = ['member_id'=>$mid, 'role'=>$role, 'is_lead'=>$isLead];
      }
    } elseif (is_array($memberIds)) {
      foreach ($memberIds as $mid) {
        if (!is_string($mid) || !preg_match('/^[a-f0-9\-]{36}$/', $mid)) continue;
        $normalized[] = ['member_id'=>$mid, 'role'=>$defaultRole, 'is_lead'=>false];
      }
    }

    db_tx(function(PDO $pdo) use ($team, $teamId, $normalized) {
      // replace current assignments
      $pdo->prepare('DELETE FROM team_member WHERE team_id=:tid')->execute([':tid'=>$teamId]);
      $ins = $pdo->prepare('INSERT INTO team_member(team_id, member_id, role_in_team, is_lead) VALUES(:tid, :mid, :r, :l)');
      foreach ($normalized as $n) {
        $mid = $n['member_id'];
        // ensure member checked in and not checked out
        $chk = $pdo->prepare('SELECT 1 FROM checkin WHERE incident_id=:iid AND member_id=:mid AND status=\'in\'');
        $chk->execute([':iid'=>$team['incident_id'], ':mid'=>$mid]);
        if (!$chk->fetchColumn()) continue;
        $ins->execute([':tid'=>$teamId, ':mid'=>$mid, ':r'=>$n['role'], ':l'=>$n['is_lead'] ? 1 : 0]);
      }
    });

    EventsController::emitIncident($team['incident_id'], null, 'all', 'teams.updated', ['incident_id'=>$team['incident_id']]);
    api_json(['ok'=>true]);
  }
}
