<?php
declare(strict_types=1);

final class TransfersController {
  public static function inbox(): void {
    $u = require_login();
    $pdo = db();

    $stmt = $pdo->prepare('SELECT t.id, t.incident_id, t.type, t.state, t.from_org_id, t.to_org_id, t.mode,
        t.payload_json, t.created_at, t.expires_at,
        o1.name AS from_org_name, o2.name AS to_org_name
      FROM transfer t
      JOIN org o1 ON o1.id=t.from_org_id
      JOIN org o2 ON o2.id=t.to_org_id
      WHERE t.to_org_id=:oid AND t.state IN (\'pending\',\'pending_fast\')
      ORDER BY t.created_at DESC LIMIT 200');
    $stmt->execute([':oid'=>$u['org_id']]);
    $rows = $stmt->fetchAll();
    foreach ($rows as &$r) {
      $r['payload'] = json_decode($r['payload_json'] ?? '{}', true);
      unset($r['payload_json']);
    }
    api_json(['ok'=>true, 'inbox'=>$rows]);
  }

  public static function offerOrder(string $incidentId): void {
    $u = require_role(['admin','el','gf']);
    $in = json_input();
    $orderId = (string)($in['order_id'] ?? '');
    $toOrgId = (string)($in['to_org_id'] ?? '');
    if (!preg_match('/^[a-f0-9\-]{36}$/', $orderId) || !preg_match('/^[a-f0-9\-]{36}$/', $toOrgId)) {
      api_error(400, 'order_id and to_org_id required');
    }

    $mode = (string)($in['mode'] ?? (public_settings()['ops']['transferModeDefault'] ?? 'offer'));
    $fastGrace = (int)(public_settings()['ops']['fastModeGraceSeconds'] ?? 120);

    $pdo = db();
    $stmt = $pdo->prepare('SELECT id, assigned_org_id, incident_id, title FROM "order" WHERE id=:id AND incident_id=:iid');
    $stmt->execute([':id'=>$orderId, ':iid'=>$incidentId]);
    $o = $stmt->fetch();
    if (!$o) api_error(404, 'order not found');

    // GF can only transfer from own org
    if (!in_array($u['role'], ['admin','el'], true) && $o['assigned_org_id'] !== $u['org_id']) {
      api_error(403, 'cannot transfer order from other org');
    }

    $state = ($mode === 'fast') ? 'pending_fast' : 'pending';
    $expiresAt = null;
    if ($mode === 'fast') {
      $expiresAt = gmdate('c', time() + $fastGrace);
    }

    $tid = null;
    db_tx(function(PDO $pdo) use ($incidentId, $u, $toOrgId, $orderId, $o, $mode, $state, $expiresAt, &$tid) {
      $stmt = $pdo->prepare('INSERT INTO transfer(id, incident_id, type, state, mode, from_org_id, to_org_id, payload_json, expires_at, created_by_user_id)
        VALUES (gen_random_uuid(), :iid, :type, :st, :mode, :from, :to, :p::jsonb, :exp, :uid) RETURNING id');
      $stmt->execute([
        ':iid'=>$incidentId,
        ':type'=>'order_transfer',
        ':st'=>$state,
        ':mode'=>$mode,
        ':from'=>$u['org_id'],
        ':to'=>$toOrgId,
        ':p'=>json_encode(['order_id'=>$orderId, 'order_title'=>$o['title']], JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES),
        ':exp'=>$expiresAt,
        ':uid'=>$u['id'],
      ]);
      $tid = (string)$stmt->fetchColumn();

      $pdo->prepare('INSERT INTO audit_log(id, incident_id, actor_user_id, action, entity_type, entity_id, meta_json)
        VALUES (gen_random_uuid(), :iid, :uid, :act, :et, :eid, :m::jsonb)')
        ->execute([':iid'=>$incidentId, ':uid'=>$u['id'], ':act'=>'transfer.offer', ':et'=>'transfer', ':eid'=>$tid, ':m'=>json_encode(['type'=>'order_transfer','order_id'=>$orderId,'to_org_id'=>$toOrgId,'mode'=>$mode])]);
    });

    // Notify target org
    EventsController::emitIncident($incidentId, $toOrgId, 'org', 'inbox.updated', ['incident_id'=>$incidentId]);
    api_json(['ok'=>true, 'id'=>$tid], 201);
  }

  public static function offerMemberLoan(string $incidentId): void {
    $u = require_role(['admin','el','gf']);
    $in = json_input();
    $memberId = (string)($in['member_id'] ?? '');
    $toOrgId = (string)($in['to_org_id'] ?? '');
    if (!preg_match('/^[a-f0-9\-]{36}$/', $memberId) || !preg_match('/^[a-f0-9\-]{36}$/', $toOrgId)) {
      api_error(400, 'member_id and to_org_id required');
    }

    // Only from-own-org unless EL/admin
    if (!in_array($u['role'], ['admin','el'], true)) {
      $chk = db()->prepare('SELECT 1 FROM member WHERE id=:id AND org_id=:oid AND is_active=true');
      $chk->execute([':id'=>$memberId, ':oid'=>$u['org_id']]);
      if (!$chk->fetchColumn()) api_error(403, 'member not in your org');
    }

    $mode = (string)($in['mode'] ?? (public_settings()['ops']['transferModeDefault'] ?? 'offer'));
    $fastGrace = (int)(public_settings()['ops']['fastModeGraceSeconds'] ?? 120);
    $state = ($mode === 'fast') ? 'pending_fast' : 'pending';
    $expiresAt = null;
    if ($mode === 'fast') $expiresAt = gmdate('c', time() + $fastGrace);

    $m = db()->prepare('SELECT id, display_name, short_code, org_id FROM member WHERE id=:id');
    $m->execute([':id'=>$memberId]);
    $row = $m->fetch();
    if (!$row) api_error(404, 'member not found');

    $tid = null;
    db_tx(function(PDO $pdo) use ($incidentId, $u, $toOrgId, $memberId, $row, $mode, $state, $expiresAt, &$tid) {
      $stmt = $pdo->prepare('INSERT INTO transfer(id, incident_id, type, state, mode, from_org_id, to_org_id, payload_json, expires_at, created_by_user_id)
        VALUES (gen_random_uuid(), :iid, :type, :st, :mode, :from, :to, :p::jsonb, :exp, :uid) RETURNING id');
      $stmt->execute([
        ':iid'=>$incidentId,
        ':type'=>'member_loan',
        ':st'=>$state,
        ':mode'=>$mode,
        ':from'=>$row['org_id'],
        ':to'=>$toOrgId,
        ':p'=>json_encode(['member_id'=>$memberId, 'member_name'=>$row['display_name'], 'member_code'=>$row['short_code']], JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES),
        ':exp'=>$expiresAt,
        ':uid'=>$u['id'],
      ]);
      $tid = (string)$stmt->fetchColumn();
    });

    EventsController::emitIncident($incidentId, $toOrgId, 'org', 'inbox.updated', ['incident_id'=>$incidentId]);
    api_json(['ok'=>true, 'id'=>$tid], 201);
  }

  public static function decide(string $transferId): void {
    $u = require_login();
    $in = json_input();
    $decision = (string)($in['decision'] ?? ''); // accept|reject
    if (!in_array($decision, ['accept','reject'], true)) api_error(400, 'decision must be accept or reject');

    db_tx(function(PDO $pdo) use ($transferId, $u, $decision) {
      $stmt = $pdo->prepare('SELECT * FROM transfer WHERE id=:id FOR UPDATE');
      $stmt->execute([':id'=>$transferId]);
      $t = $stmt->fetch();
      if (!$t) api_error(404, 'transfer not found');
      if ($t['to_org_id'] !== $u['org_id'] && $u['role'] !== 'admin') api_error(403, 'not your inbox');
      if (!in_array($t['state'], ['pending','pending_fast'], true)) api_error(400, 'already decided');

      $newState = ($decision === 'accept') ? 'accepted' : 'rejected';

      // Apply effects on accept
      $payload = json_decode($t['payload_json'] ?? '{}', true) ?: [];
      if ($decision === 'accept') {
        if ($t['type'] === 'order_transfer') {
          $orderId = (string)($payload['order_id'] ?? '');
          if ($orderId) {
            $pdo->prepare('UPDATE "order" SET assigned_org_id=:to, assigned_team_id=NULL, updated_at=now() WHERE id=:oid')
              ->execute([':to'=>$t['to_org_id'], ':oid'=>$orderId]);
          }
        }
        if ($t['type'] === 'member_loan') {
          $memberId = (string)($payload['member_id'] ?? '');
          if ($memberId) {
            $pdo->prepare('INSERT INTO member_loan(id, incident_id, member_id, from_org_id, to_org_id, transfer_id, status)
              VALUES (gen_random_uuid(), :iid, :mid, :from, :to, :tid, :st)')
              ->execute([':iid'=>$t['incident_id'], ':mid'=>$memberId, ':from'=>$t['from_org_id'], ':to'=>$t['to_org_id'], ':tid'=>$t['id'], ':st'=>'active']);
          }
        }
      }

      $pdo->prepare('UPDATE transfer SET state=:st, decided_at=now(), decided_by_user_id=:uid WHERE id=:id')
        ->execute([':st'=>$newState, ':uid'=>$u['id'], ':id'=>$transferId]);

      $pdo->prepare('INSERT INTO audit_log(id, incident_id, actor_user_id, action, entity_type, entity_id, meta_json)
        VALUES (gen_random_uuid(), :iid, :uid, :act, :et, :eid, :m::jsonb)')
        ->execute([':iid'=>$t['incident_id'], ':uid'=>$u['id'], ':act'=>'transfer.decide', ':et'=>'transfer', ':eid'=>$transferId, ':m'=>json_encode(['decision'=>$decision,'type'=>$t['type']])]);

      // notify both orgs
      EventsController::emitIncident((string)$t['incident_id'], (string)$t['to_org_id'], 'org', 'inbox.updated', ['incident_id'=>$t['incident_id']]);
      EventsController::emitIncident((string)$t['incident_id'], (string)$t['from_org_id'], 'org', 'transfer.decided', ['incident_id'=>$t['incident_id'], 'transfer_id'=>$transferId, 'state'=>$newState]);
      EventsController::emitIncident((string)$t['incident_id'], null, 'all', 'orders.updated', ['incident_id'=>$t['incident_id']]);
      EventsController::emitIncident((string)$t['incident_id'], null, 'all', 'teams.updated', ['incident_id'=>$t['incident_id']]);
      EventsController::emitIncident((string)$t['incident_id'], null, 'all', 'participants.updated', ['incident_id'=>$t['incident_id']]);
    });

    api_json(['ok'=>true]);
  }
}
