<?php
declare(strict_types=1);

require_once __DIR__ . '/db.php';
require_once __DIR__ . '/respond.php';
require_once __DIR__ . '/settings.php';

function ensure_session(): void {
  if (session_status() !== PHP_SESSION_ACTIVE) {
    $c = cfg();
    $https = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') || (isset($_SERVER['HTTP_X_FORWARDED_PROTO']) && $_SERVER['HTTP_X_FORWARDED_PROTO'] === 'https');
    if (($c['force_https'] ?? false) && !$https) {
      api_error(400, 'HTTPS required');
    }
    session_set_cookie_params([
      'httponly' => true,
      'secure' => $https,
      'samesite' => 'Lax',
      'path' => '/',
    ]);
    session_start();
  }
}

function csrf_token(): string {
  ensure_session();
  if (empty($_SESSION['csrf'])) {
    $_SESSION['csrf'] = bin2hex(random_bytes(32));
  }
  return $_SESSION['csrf'];
}

function csrf_check(): void {
  // Require for state-changing methods.
  $m = $_SERVER['REQUEST_METHOD'] ?? 'GET';
  if (in_array($m, ['POST','PUT','PATCH','DELETE'], true)) {
    $hdr = $_SERVER['HTTP_X_CSRF_TOKEN'] ?? '';
    if (!$hdr || !hash_equals(csrf_token(), $hdr)) {
      api_error(403, 'CSRF check failed');
    }
  }
}

function current_user(): ?array {
  ensure_session();
  if (empty($_SESSION['uid'])) return null;
  $uid = (int)$_SESSION['uid'];
  $stmt = db()->prepare('SELECT id, username, role, org_id, display_name, is_active FROM app_user WHERE id = :id');
  $stmt->execute([':id'=>$uid]);
  $u = $stmt->fetch();
  if (!$u || !$u['is_active']) return null;
  return $u;
}

function require_login(): array {
  $u = current_user();
  if (!$u) api_error(401, 'Not logged in');
  return $u;
}

function require_role(array $roles): array {
  $u = require_login();
  if (!in_array($u['role'], $roles, true)) {
    api_error(403, 'Forbidden');
  }
  return $u;
}

function password_hash_app(string $password): string {
  return password_hash($password, PASSWORD_DEFAULT);
}

function login(string $username, string $password): array {
  ensure_session();
  $stmt = db()->prepare('SELECT id, username, role, org_id, display_name, password_hash, is_active FROM app_user WHERE username = :u');
  $stmt->execute([':u'=>$username]);
  $row = $stmt->fetch();
  if (!$row || !$row['is_active'] || !password_verify($password, $row['password_hash'])) {
    api_error(401, 'Invalid credentials');
  }
  $_SESSION['uid'] = (int)$row['id'];
  $_SESSION['csrf'] = bin2hex(random_bytes(32));
  // Update last seen
  db()->prepare('UPDATE app_user SET last_seen_at = now() WHERE id = :id')->execute([':id'=>$row['id']]);
  unset($row['password_hash']);
  $row['csrf'] = $_SESSION['csrf'];
  $row['settings'] = public_settings();
  return $row;
}

function logout(): void {
  ensure_session();
  $_SESSION = [];
  if (ini_get('session.use_cookies')) {
    $params = session_get_cookie_params();
    setcookie(session_name(), '', time() - 42000,
      $params['path'], $params['domain'], $params['secure'], $params['httponly']
    );
  }
  session_destroy();
}
